<?php

/**
 * StatusPage Updater.
 *
 * Handles automatic updates from GitHub.
 *
 * @package StatusPageWidget
 */

// Exit if accessed directly.
if (! defined('ABSPATH')) {
    exit;
}

/**
 * Class StatusPage_Updater
 *
 * Implements GitHub-based automatic updates.
 */
class StatusPage_Updater
{

    /**
     * GitHub repository owner.
     *
     * @var string
     */
    const GITHUB_OWNER = 'hosted-status-page';

    /**
     * GitHub repository name.
     *
     * @var string
     */
    const GITHUB_REPO = 'wordpress-status-page-widget';

    /**
     * Plugin basename.
     *
     * @var string
     */
    private static $plugin_basename;

    /**
     * Cached GitHub release data.
     *
     * @var array|null
     */
    private static $github_release;

    /**
     * Initialize the updater.
     */
    public static function init()
    {
        self::$plugin_basename = plugin_basename(STATUSPAGE_WIDGET_PATH . 'status-page-widget.php');

        // Hook into the update system.
        add_filter('pre_set_site_transient_update_plugins', array(__CLASS__, 'check_for_update'));
        add_filter('plugins_api', array(__CLASS__, 'plugin_info'), 20, 3);
        add_filter('upgrader_source_selection', array(__CLASS__, 'fix_directory_name'), 10, 4);

        // Add GitHub link to plugin row.
        add_filter('plugin_row_meta', array(__CLASS__, 'plugin_row_meta'), 10, 2);
    }

    /**
     * Check for updates from GitHub.
     *
     * @param object $transient The update transient.
     * @return object Modified transient.
     */
    public static function check_for_update($transient)
    {
        if (empty($transient->checked)) {
            return $transient;
        }

        $release = self::get_latest_release();

        if (! $release) {
            return $transient;
        }

        $new_version = ltrim($release['tag_name'], 'v');

        if (version_compare(STATUSPAGE_WIDGET_VERSION, $new_version, '<')) {
            $plugin_info = array(
                'slug'        => 'status-page-widget',
                'plugin'      => self::$plugin_basename,
                'new_version' => $new_version,
                'url'         => $release['html_url'],
                'package'     => $release['zipball_url'],
                'icons'       => array(
                    '1x' => STATUSPAGE_API_BASE . '/static/img/icon-128.png',
                    '2x' => STATUSPAGE_API_BASE . '/static/img/icon-256.png',
                ),
                'banners'     => array(
                    'low'  => STATUSPAGE_API_BASE . '/static/img/banner-772x250.png',
                    'high' => STATUSPAGE_API_BASE . '/static/img/banner-1544x500.png',
                ),
            );

            $transient->response[self::$plugin_basename] = (object) $plugin_info;
        } else {
            // No update available.
            $transient->no_update[self::$plugin_basename] = (object) array(
                'slug'        => 'status-page-widget',
                'plugin'      => self::$plugin_basename,
                'new_version' => STATUSPAGE_WIDGET_VERSION,
                'url'         => 'https://github.com/' . self::GITHUB_OWNER . '/' . self::GITHUB_REPO,
            );
        }

        return $transient;
    }

    /**
     * Provide plugin info for the WordPress plugins API.
     *
     * @param false|object|array $result The result object or array.
     * @param string             $action The type of information being requested.
     * @param object             $args   Plugin API arguments.
     * @return false|object Plugin info or false.
     */
    public static function plugin_info($result, $action, $args)
    {
        if ('plugin_information' !== $action) {
            return $result;
        }

        if (! isset($args->slug) || 'status-page-widget' !== $args->slug) {
            return $result;
        }

        $release = self::get_latest_release();

        if (! $release) {
            return $result;
        }

        $new_version = ltrim($release['tag_name'], 'v');

        $plugin_info = array(
            'name'              => 'StatusPage Widget',
            'slug'              => 'status-page-widget',
            'version'           => $new_version,
            'author'            => '<a href="https://statuspage.me">StatusPage.me</a>',
            'author_profile'    => 'https://statuspage.me',
            'homepage'          => 'https://github.com/' . self::GITHUB_OWNER . '/' . self::GITHUB_REPO,
            'requires'          => '5.8',
            'tested'            => '6.4',
            'requires_php'      => '7.4',
            'sections'          => array(
                'description'  => 'Display real-time status page widgets from StatusPage.me on your WordPress site. Includes Gutenberg block, classic widget, and shortcodes.',
                'installation' => 'Upload the plugin files to the `/wp-content/plugins/status-page-widget` directory, or install the plugin through the WordPress plugins screen directly. Activate the plugin through the \'Plugins\' screen in WordPress.',
                'changelog'    => isset($release['body']) ? $release['body'] : '',
            ),
            'download_link'     => $release['zipball_url'],
            'banners'           => array(
                'low'  => STATUSPAGE_API_BASE . '/static/img/banner-772x250.png',
                'high' => STATUSPAGE_API_BASE . '/static/img/banner-1544x500.png',
            ),
            'icons'             => array(
                '1x' => STATUSPAGE_API_BASE . '/static/img/icon-128.png',
                '2x' => STATUSPAGE_API_BASE . '/static/img/icon-256.png',
            ),
        );

        return (object) $plugin_info;
    }

    /**
     * Fix the directory name after extracting the zip.
     *
     * GitHub zips have directory names like "owner-repo-hash", but we need "status-page-widget".
     *
     * @param string      $source        Path to the extracted source.
     * @param string      $remote_source Remote file source.
     * @param WP_Upgrader $upgrader      Upgrader instance.
     * @param array       $hook_extra    Extra arguments passed to the upgrader.
     * @return string|WP_Error New source path or error.
     */
    public static function fix_directory_name($source, $remote_source, $upgrader, $hook_extra)
    {
        global $wp_filesystem;

        // Only process our plugin.
        if (! isset($hook_extra['plugin']) || self::$plugin_basename !== $hook_extra['plugin']) {
            return $source;
        }

        // Check if source directory starts with GitHub-style naming.
        $source_base = basename($source);
        if (strpos($source_base, self::GITHUB_OWNER . '-' . self::GITHUB_REPO) !== 0) {
            return $source;
        }

        // Rename to expected directory.
        $new_source = trailingslashit(dirname($source)) . 'status-page-widget/';

        if ($wp_filesystem->move($source, $new_source)) {
            return $new_source;
        }

        return new WP_Error(
            'rename_failed',
            __('Unable to rename the plugin directory.', 'status-page-widget')
        );
    }

    /**
     * Add GitHub link to plugin row meta.
     *
     * @param array  $links Plugin row meta links.
     * @param string $file  Plugin file.
     * @return array Modified links.
     */
    public static function plugin_row_meta($links, $file)
    {
        if (self::$plugin_basename === $file) {
            $links[] = sprintf(
                '<a href="%s" target="_blank">%s</a>',
                'https://github.com/' . self::GITHUB_OWNER . '/' . self::GITHUB_REPO,
                __('GitHub', 'status-page-widget')
            );
        }

        return $links;
    }

    /**
     * Get the latest release from GitHub.
     *
     * @return array|null Release data or null on failure.
     */
    private static function get_latest_release()
    {
        if (null !== self::$github_release) {
            return self::$github_release;
        }

        // Check cache first.
        $cached = get_transient('statuspage_widget_github_release');
        if (false !== $cached) {
            self::$github_release = $cached;
            return $cached;
        }

        $api_url = sprintf(
            'https://api.github.com/repos/%s/%s/releases/latest',
            self::GITHUB_OWNER,
            self::GITHUB_REPO
        );

        $response = wp_remote_get(
            $api_url,
            array(
                'timeout'    => 10,
                'user-agent' => 'StatusPage-Widget/' . STATUSPAGE_WIDGET_VERSION . ' WordPress/' . get_bloginfo('version'),
                'headers'    => array(
                    'Accept' => 'application/vnd.github.v3+json',
                ),
            )
        );

        if (is_wp_error($response) || 200 !== wp_remote_retrieve_response_code($response)) {
            // Cache failure to prevent repeated requests.
            set_transient('statuspage_widget_github_release', array(), HOUR_IN_SECONDS);
            self::$github_release = array();
            return null;
        }

        $body    = wp_remote_retrieve_body($response);
        $release = json_decode($body, true);

        if (json_last_error() !== JSON_ERROR_NONE || empty($release['tag_name'])) {
            set_transient('statuspage_widget_github_release', array(), HOUR_IN_SECONDS);
            self::$github_release = array();
            return null;
        }

        // Cache for 12 hours.
        set_transient('statuspage_widget_github_release', $release, 12 * HOUR_IN_SECONDS);
        self::$github_release = $release;

        return $release;
    }
}
